package data;

import android.util.Log;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.topjohnwu.superuser.Shell;
import java.io.File;
import java.util.List;

public final class PropertiesManager {

    private static final String TAG = "PropertiesManager";
    private static final String FILE_NAME = "properties.json";
    private static final String PACKAGE_NAME = "com.crjase.systemuihook"; // Replace with your actual package name
    private static final String COMMON_DATA_DIR = "/data/misc/" + PACKAGE_NAME;
    private static final String FILE_PATH = COMMON_DATA_DIR + File.separator + FILE_NAME;
    private static final Gson gson = new GsonBuilder().setPrettyPrinting().create();
    private static UIProperties properties;

    // Private constructor to prevent instantiation
    private PropertiesManager() {
        Shell.getShell(); // Ensure a root shell is available
        loadProperties();
    }

    /**
     * Initializes the PropertiesManager, ensuring a root shell and loading properties.
     */
    public static void initialize() {
        // This method can be called once from your Xposed module's onPackageLoaded
        // to set everything up.
        if (properties == null) {
            new PropertiesManager();
        }
    }

    /**
     * Loads properties from the properties.json file using a root shell.
     */
    private static void loadProperties() {
        // Check if the directory exists and create it if not
        Shell.cmd("mkdir -p " + COMMON_DATA_DIR).exec();

        List<String> checkFileResult = Shell.cmd("ls " + FILE_PATH).exec().getOut();

        if (checkFileResult.isEmpty()) {
            Log.d(TAG, "properties.json not found. Creating a new one with default values.");
            properties = new UIProperties();
            saveProperties();
        } else {
            List<String> fileContent = Shell.cmd("cat " + FILE_PATH).exec().getOut();
            StringBuilder jsonContent = new StringBuilder();
            for (String line : fileContent) {
                jsonContent.append(line);
            }

            try {
                properties = gson.fromJson(jsonContent.toString(), UIProperties.class);
                if (properties == null) {
                    Log.e(TAG, "Failed to parse properties.json. Creating a new one.");
                    properties = new UIProperties();
                    saveProperties();
                }
            } catch (Exception e) {
                Log.e(TAG, "Error loading properties file: " + e.getMessage());
                properties = new UIProperties();
                saveProperties();
            }
        }
    }

    /**
     * Saves the current properties to the properties.json file using a root shell.
     */
    public static void saveProperties() {
        String json = gson.toJson(properties);
        String command = "echo '" + json + "' | tee " + FILE_PATH;
        Shell.cmd(command).exec();
        Log.d(TAG, "Properties saved successfully to " + FILE_PATH);
    }

    /**
     * Returns the UIProperties object.
     * @return The current UI properties.
     */
    public static UIProperties getProperties() {
        if (properties == null) {
            initialize();
        }
        return properties;
    }

    /**
     * Updates the properties with new values and saves the changes.
     * @param newProperties The new UIProperties object.
     */
    public static void updateProperties(UIProperties newProperties) {
        properties = newProperties;
        saveProperties();
    }
}